<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\InvoiceModel;
use App\Models\PaymentsModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class MobInvoicePaymentController extends Controller
{
 public function getInvoicesWithoutPayments(Request $request)
    {
        $invoices = InvoiceModel::with([
            'client.user:id,name,phone',
            'createdBy:id,name',
            'invoiceItems:id,invoice_id,product_id,product_name,quantity,price,margin_price,total',
            'invoiceItems.invoiceItemTaxes:id,invoice_item_id,tax_id,tax',
        ])
            ->whereDoesntHave('payments')
            ->select([
                'id',
                'invoice_id',
                'client_id',
                'branch_id',
                'invoice_date',
                'due_date',
                'gst',
                'amount',
                'final_amount',
                'discount_type',
                'discount',
                'note',
                'term',
                'status',
                'created_by',
                'created_at',
                'updated_at',
            ])
            ->where('branch_id', $request->branch_id)
            ->where('oprntl_flag', 'A')
            ->get();
   
        return response()->json([
            'invoices' => $invoices
        ]);
    }

    public function getInvoicesWithPayments(Request $request)
{
    $query = InvoiceModel::with([
            'client.user:id,name,phone',
            'createdBy:id,name',
            'invoiceItems:id,invoice_id,product_id,product_name,quantity,price,margin_price,total',
            'invoiceItems.invoiceItemTaxes:id,invoice_item_id,tax_id,tax',
        ])
        ->whereHas('payments')
        ->select([
            'id',
            'invoice_id',
            'client_id',
            'branch_id',
            'invoice_date',
            'due_date',
            'gst',
            'amount',
            'final_amount',
            'discount_type',
            'discount',
            'note',
            'term',
            'status',
            'created_by',
            'created_at',
            'updated_at',
        ])
        ->where('branch_id', $request->branch_id)
        ->where('oprntl_flag', 'A');
        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('invoice_id', 'like', "%$searchTerm%")
                    ->orWhereHas('client.user', function ($q2) use ($searchTerm) {
                        $q2->where('name', 'like', "%$searchTerm%")
                            ->orWhere('phone', 'like', "%$searchTerm%");
                    });
            });
        }

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->whereBetween('invoice_date', [
                $request->from_date,
                $request->to_date
            ]);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $perPage = $request->get('per_page', 10);
        $invoices = $query->orderBy('id', 'desc')->paginate($perPage);

    return response()->json([
        'invoices' => $invoices
    ]);
}

    public function store(Request $request)
    {
       
        $validated = $request->validate([
            'invoice_id' => 'required',
            'total_amount' => 'required',
            'amount' => 'required',
            'payment_date' => 'required',
            'payment_mode' => 'required',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            // Create payment
            $payment = PaymentsModel::create($validated);

           $invoice = InvoiceModel::where('id', $validated['invoice_id'])->first();
            $this->updateInvoiceStatus($invoice);

            DB::commit();

            return response()->json([
                'success' => true,
                'data' => $payment,
                'message' => 'Payment created successfully',
            ], 201);
        } catch (\Exception $e) {
             Log::info($e);
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function show(PaymentsModel $payment)
    {
        $payment->load(['invoice.client.user', 'mode']);
        
        return response()->json([
            'success' => true,
            'data' => $payment,
        ]);
    }

    public function update(Request $request, PaymentsModel $payment)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_mode' => 'required|in:1,2,3',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $payment->update($validated);

            $invoice = $payment->invoice;
            $this->updateInvoiceStatus($invoice);

            DB::commit();

            return response()->json([
                'success' => true,
                'data' => $payment,
                'message' => 'Payment updated successfully',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payment',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function destroy(PaymentsModel $payment)
    {
        DB::beginTransaction();
        try {
            $payment->delete();

            // Update invoice status if needed
            $invoice = $payment->invoice;
            $this->updateInvoiceStatus($invoice);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment deleted successfully',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete payment',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function getInvoiceDetails(InvoiceModel $invoice)
    {
        $invoice->load(['client.user', 'payments']);
        
        $totalAmount = $invoice->final_amount;
        $paidAmount = $invoice->payments->sum('amount');
        $dueAmount = $totalAmount - $paidAmount;
        
        return response()->json([
            'success' => true,
            'data' => [
                'totalAmount' => $totalAmount,
                'paidAmount' => $paidAmount,
                'dueAmount' => $dueAmount,
                'invoice' => $invoice,
            ],
        ]);
    }

    private function updateInvoiceStatus(InvoiceModel $invoice)
    {
        $totalAmount = $invoice->final_amount;
        $paidAmount = $invoice->payments->sum('amount');
        
        if ($paidAmount >= $totalAmount) {
            $invoice->status = 2; 
        } elseif ($paidAmount > 0) {
            $invoice->status = 3; 
        } else {
            $invoice->status = 1;
        }
        
        $invoice->save();
    }
}
