<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\InvoiceItemModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductHistoryModel;
use App\Models\ProductModel;
use App\Models\ProductPaymentModel;
use App\Models\QuotesItemsModel;
use App\Models\TaxesModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Log;

class MobProductController extends Controller
{

    public function index(Request $request)
    {
        $query = ProductModel::with([
            'category:id,name',
            'supplier:id,supplier_name'
        ])->where('branch_id',$request->branch_id);

        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                    ->orWhere('code', 'like', "%{$searchTerm}%")
                    ->orWhere('hsn_code', 'like', "%{$searchTerm}%")
                    ->orWhereHas('category', function ($q) use ($searchTerm) {
                        $q->where('name', 'like', "%{$searchTerm}%");
                    })
                    ->orWhereHas('supplier', function ($q) use ($searchTerm) {
                        $q->where('supplier_name', 'like', "%{$searchTerm}%");
                    });
            });
        }

        $perPage = $request->per_page ?? 10;
        $page = $request->page ?? 1;

        $products = $query->paginate($perPage, ['*'], 'page', $page);

        // Return just the data array
        return response()->json($products->items());
    }

        public function allProducts($branchId)
        {
            $products = ProductModel::where('branch_id',$branchId)->get();
            return response()->json($products);
        }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'code' => 'required|unique:products,code',
            'unit_price' => 'required|numeric',
            'margin_price' => 'required|numeric',
            'category_id' => 'required|integer',
            'supplier_id' => 'required|integer',
            'qty' => 'required|integer',
        ]);

        DB::beginTransaction();

        try {
            $productPath = null;
            if ($request->hasFile('product_image')) {
                $productPath = storeFile($request->file('product_image'), 'products');
            }

            $status = 1;
            $upi = $request->input('upi', 0);
            $online = $request->input('online', 0);
            $cash = $request->input('cash', 0);
            $totalAmount = $request->input('total_amount', 0);

            if ($upi || $online || $cash) {
                $status = ($totalAmount == ($upi + $online + $cash)) ? 2 : 3;
            }

            $product = ProductModel::create([
                'name' => $request->name,
                'code' => $request->code,
                'hsn_code' => $request->hsn_code,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'margin_price' => $request->margin_price,
                'description' => $request->description,
                'products_image' => $productPath,
                'branch_id' => $request->branch_id,
                'qty' => $request->qty,
                'total_qty' => $request->qty,
            ]);

            $productDetail = ProductDetailsModel::create([
                'product_id' => $product->id,
                'name' => $request->name,
                'code' => $request->code,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'margin_price' => $request->margin_price,
                'description' => $request->description,
                'products_image' => $productPath,
                'branch_id' => $request->branch_id,
                'qty' => $request->qty,
                'total_amount' => $totalAmount,
                'status' => $status,
            ]);

            // Handle nested payments array (from mobile)
            if ($request->has('payments')) {
                foreach ($request->payments as $payment) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $productDetail->product_details_sid,
                        'payment_type' => $payment['payment_type'],
                        'payment_date' => $payment['payment_date'],
                        'total_amount' => $payment['total_amount'],
                        'paid_amount' => $payment['paid_amount'],
                    ]);
                }
            }

            ProductHistoryModel::create([
                'product_id' => $product->id,
                'branch_id' => $request->branch_id,
                'name' => $request->name,
                'code' => $request->code,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'qty' => $request->qty,
                'description' => $request->description,
                'product_image' => $productPath,
                'action' => 'A',
                'created_by' => Auth::user()->id,
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Product added successfully!',
                'data' => $product->load('category', 'supplier'),
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Product store error: ' . $e->getMessage());

            return response()->json([
                'message' => 'Something went wrong.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        return ProductModel::findOrFail($id);
    }

    public function update(Request $request, $id)
    {
        $product = ProductModel::findOrFail($id);
        $product->update($request->all());
        return response()->json($product, 200);
    }

    public function destroy($id)
    {
       DB::beginTransaction();

        try {
            $product = ProductModel::with(['details.payments'])
                ->where('active_flag', 'A')
                ->find($id);

            if (!$product) {
                return response()->json( ['message' => 'Product not found!'],404);
            }

            $inInvoice = InvoiceItemModel::where('product_id', $product->id)
                ->where('active_flag', 'A')
                ->exists();

            $inQuotes = QuotesItemsModel::where('product_id', $product->id)
                ->where('active_flag', 'A')
                ->exists();

            $inPayment = ProductPaymentModel::whereIn('product_dtl_id', $product->details->pluck('product_details_sid'))
                ->where('active_flag', 'A')
                ->exists();

            if ($inInvoice || $inQuotes || $inPayment) {
                DB::rollBack();
                return response()->json(['message' => 'Cannot delete product: It exists in active invoices, quotes, or payments!'], 400);
            }

            // Create history record
            ProductHistoryModel::create([
                'product_id' => $product->id,
                'branch_id' => $product->branch_id,
                'name' => $product->name,
                'code' => $product->code,
                'category_id' => $product->category_id,
                'supplier_id' => $product->supplier_id,
                'unit_price' => $product->unit_price,
                'qty' => $product->qty,
                'description' => $product->description,
                'product_image' => $product->products_image,
                'action' => 'D',
                'created_by' => Auth::user()->id,
            ]);

            foreach ($product->details as $detail) {
                if ($detail->payments->isNotEmpty()) {
                    $detail->payments->delete();
                }
                $detail->delete();
            }

            $product->delete();

            DB::commit();

            return response()->json(null, 204);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Product deletion failed for ID {$id}: " . $e->getMessage());
            return response()->json(['message' => $e->getMessage()], 400);
        }
       
    }

}
