@extends('layouts.layoutapp')

@section('content')
    <div class="row">
        <div class="col-12">
            <div class="card me-2">
                <form action="{{ route('product-update') }}" method="POST" enctype="multipart/form-data">
                    @csrf
                    <div class="card-header pb-0">
                        <div class="d-flex flex-row justify-content-between">
                            <div>
                                <h5 class="mb-0">Edit Product</h5>
                            </div>
                            <div>
                                <button type="submit" class="btn btn-primary btn-sm">Update Product</button>
                                <a href="{{ route('product-index') }}" class="btn btn-secondary btn-sm">Cancel</a>
                            </div>
                        </div>
                    </div>

                    <div class="card-body">
                        <input type="hidden" name="product_id" value="{{$product->id}}">
                        <div class="row">
                            <!-- Left Column -->
                            <div class="col-lg-6">
                                <div class="row">
                                    <div class="col-lg-6 mb-3">
                                        <label for="name" class="form-label">Name:</label>
                                        <input type="text" id="name" name="name" class="form-control"
                                            value="{{ old('name', $product->name) }}" placeholder="Enter product name"
                                            required>
                                        @error('name') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>
                                    <div class="col-lg-6 mb-3">
                                        <label for="native_name" class="form-label">Native Name:</label>
                                        <input type="text" id="native_name" name="native_name" class="form-control"
                                            value="{{ old('native_name', $product->native_name) }}"
                                            placeholder="Enter native name" required>
                                        @error('name') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>

                                    <div class="col-lg-6 mb-3">
                                        <label for="margin_price" class="form-label">Buying Price / Unit:</label>
                                        <input type="number" id="margin_price" name="margin_price" class="form-control"
                                            value="{{ old('margin_price', $product->margin_price) }}" placeholder="0.00"
                                            min="0" step="0.01" required>
                                        @error('margin_price') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>
                                    <div class="col-lg-6 mb-3">
                                        <label for="unit_price" class="form-label">Selling Price / Unit:</label>
                                        <input type="number" id="unit_price" name="unit_price" class="form-control"
                                            value="{{ old('unit_price', $product->unit_price) }}" placeholder="0.00" min="0"
                                            step="0.01" required>
                                        @error('unit_price') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>
                                    <div class="col-lg-6 mb-3">
                                        <label for="supplier_id" class="form-label">Supplier:</label>
                                        <select name="supplier_id" class="form-select" required>
                                            <option value="" disabled>Select Supplier</option>
                                            <option value="" {{ empty(old('supplier_id', $product->supplier_id)) ? 'selected' : '' }}>Internal</option>
                                            @foreach ($suppliers as $supplier)
                                                <option value="{{ $supplier->id }}" {{ old('supplier_id', $product->supplier_id) == $supplier->id ? 'selected' : '' }}>
                                                    {{ $supplier->supplier_name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('supplier_id') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>

                                    <div class="col-lg-6 mb-3">
                                        <label for="description" class="form-label">Description:</label>
                                        <textarea id="description" name="description" class="form-control" rows="1"
                                            placeholder="Enter product description (optional)">{{ old('description', $product->description) }}</textarea>
                                        @error('description') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Right Column -->
                            <div class="col-lg-6">
                                <div class="row">
                                    <div class="col-lg-6 mb-3">
                                        <label for="productCode" class="form-label">Product Code:</label>
                                        <input type="text" id="productCode" name="productCode" class="form-control"
                                            value="{{ old('productCode', $product->code) }}"
                                            placeholder="Enter product code" required>
                                        @error('productCode') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>
                                    <div class="col-lg-6 mb-3">
                                        <label for="hsnCode" class="form-label">HSN Code:</label>
                                        <input type="text" id="hsnCode" name="hsnCode" class="form-control"
                                            value="{{ old('hsnCode', $product->hsn_code) }}" placeholder="Enter HSN code">
                                        @error('hsnCode') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>
                                    <div class="col-lg-6 mb-3">
                                        <label for="qty" class="form-label">Quantity:</label>
                                        <input type="number" id="qty" name="qty" class="form-control"
                                            value="{{ old('qty', $product->qty) }}" readonly placeholder="0" min="0"
                                            step="0.01" required>
                                        @error('qty') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>

                                    <div class="col-lg-6 mb-3">
                                        <label for="category_id" class="form-label">Category:</label>
                                        <select name="category_id" class="form-select" required>
                                            <option value="" selected disabled>Select Category</option>
                                            @foreach ($categories as $category)
                                                <option value="{{ $category->id }}" {{ (old('category_id', $product->category_id) == $category->id) ? 'selected' : '' }}>
                                                    {{ $category->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('category_id') <span class="text-danger">{{ $message }}</span> @enderror
                                    </div>
                                    <div class="col-lg-6 mb-3">
                                        <label for="product_image" class="form-label">Upload Image:</label>
                                        <div class="avatar-edit">
                                            <input type="file" id="product_image" name="product_image"
                                                class="form-control file-upload-input" accept="image/*"
                                                onchange="previewImage(this)">
                                            <small class="text-muted">Upload new image (max 2MB)</small>
                                        </div>
                                    </div>
                                    <div class="col-lg-6 mb-3">
                                        <div class="d-flex align-items-center">
                                            <div class="avatar-preview">
                                                <img id="imagePreview"
                                                    src="{{ $product->products_image ? getFileUrl($product->products_image) : asset('assets/img/logos/Product.png') }}"
                                                    alt="Product Image" class="img-fluid"
                                                    style="width: 80px; height: 80px; object-fit: cover;">
                                            </div>
                                        </div>
                                        @error('product_image')
                                            <span class="text-danger">{{ $message }}</span>
                                        @enderror
                                    </div>

                                </div>
                            </div>
                        </div>
                    </div>
                </form>

                <!-- Product Details Table -->
                <div class="table-responsive">
                    <table class="table table-striped table-bordered">
                        <thead>
                            <tr>
                                <th class="text-center text-xs">#</th>
                                <th class="text-center text-xs">Action</th>
                                <th class="text-center text-xs">Product Details ID</th>
                                <th class="text-center text-xs">Supplier</th>
                                <th class="text-center text-xs">Selling Price</th>
                                <th class="text-center text-xs">Buying Price</th>
                                <th class="text-center text-xs">Updated Qty</th>
                                <th class="text-center text-xs">Total Amount</th>
                                <th class="text-center text-xs">Payment Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse ($product->details as $index => $data)
                                <tr>
                                    <td class="text-center text-xs">{{ $index + 1 }}</td>
                                    <td class="text-center text-xs">
                                        @if (empty($data->assembled_item_id))
                                            <a href="javascript:void(0)" class="edit-details" id="edit_details"
                                                data-bs-toggle="modal" data-bs-target="#productDetails" data-details='@json($data)'>
                                                <i class="fa-solid fa-pen-to-square ms-2" data-bs-toggle="tooltip"
                                                    data-bs-placement="top" title="Edit Details"></i>
                                            </a>
                                        @endif
                                        <a href="javascript:void(0)" class="view-refund" data-bs-toggle="modal"
                                            data-bs-target="#productRefundDetails" data-details='@json($data)'>
                                            <i class="fa-solid fa-rotate-left ms-2" data-bs-toggle="tooltip"
                                                data-bs-placement="top" title="View Refund Details"></i>
                                        </a>
                                    </td>

                                    <td class="text-center text-xs">{{ $data->product_details_code ?? '' }}</td>
                                    <td class="text-center text-xs">
                                        {{ isset($data->suppliers) ? $data->suppliers->supplier_name : 'Internal' }}
                                    </td>
                                    <td class="text-center text-xs">{{ number_format($data->unit_price ?? '', 2) }}</td>
                                    <td class="text-center text-xs">{{ number_format($data->margin_price ?? '', 2) }}</td>
                                    <td class="text-center text-xs">{{ $data->qty ?? '' }}</td>
                                    <td class="text-center text-xs">{{ $data->total_amount ?? '' }}</td>
                                    <td class="text-center text-xs">
                                        @if ($data->paymentStatus->name == 'Paid')
                                            <span class="badge bg-success">Paid</span>
                                        @elseif ($data->paymentStatus->name == 'Partially Paid')
                                            <span class="badge bg-warning">Partially Paid</span>
                                        @else
                                            <span class="badge bg-danger">Unpaid</span>
                                        @endif
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td class="text-center" colspan="8"> No Details Found</td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Product Details Modal -->
    <div class="modal fade" id="productDetails" data-bs-keyboard="false" tabindex="-1" aria-labelledby="staticBackdropLabel"
        aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="staticBackdropLabel">Product Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form class="product-details-form">
                        @csrf
                        <input type="hidden" class="detail_id" name="detail_id" value="">
                        <input type="hidden" class="detail_code" name="detail_code" value="">
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Supplier Name & Number</label>
                                <input type="text" id="supplier_dtls" name="supplier_dtls" class="form-control"
                                    value="{{ old('supplier_dtls') }}" placeholder="Enter Supplier Details" readonly
                                    required>
                                @error('supplier_dtls') <span class="text-danger">{{ $message }}</span> @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="status" class="form-label">Reason:</label>
                                <select name="status" class="form-select" required>
                                    <option value="" selected disabled>Select Reason</option>
                                    <option value="update">Update Quantity</option>
                                    <option value="refund">Return to Supplier</option>
                                    <option value="non_returnable">Non-Returnable (Damaged)</option>
                                </select>
                                @error('status') <span class="text-danger">{{ $message }}</span> @enderror
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Purchased Quantity:</label>
                                <input type="number" id="purchased_qty" name="purchased_qty" class="form-control"
                                    value="{{ old('purchased_qty') }}" min="1" placeholder="Enter purchased quantity"
                                    required>
                                @error('purchased_qty') <span class="text-danger">{{ $message }}</span> @enderror
                            </div>
                        </div>
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped">
                                <thead>
                                    <tr>
                                        <th class="text-center">Product Qty</th>
                                        <th class="text-center">Buying Price</th>
                                        <th class="text-center">Paid Amount</th>
                                        <th class="text-center">Updated Qty</th>
                                        <th class="text-center">Amount</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td class="text-center product_qty"></td>
                                        <td class="text-center margin_price"></td>
                                        <td class="text-center paid_amount"></td>
                                        <td class="text-center updated_qty"></td>
                                        <td class="text-center"> <input type="text" name="amount"
                                                class="form-control amount" id="amount" placeholder="Enter amount" readonly>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-primary details-save">Save</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Refund Details Modal -->
    <div class="modal fade" id="productRefundDetails" data-bs-keyboard="false" tabindex="-1"
        aria-labelledby="staticBackdropLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="staticBackdropLabel">Refund Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <h4 class="product-name"></h4>
                    <div class="table-responsive">
                        <table class="table table-striped table-bordered">
                            <thead>
                                <tr>
                                    <th class="text-center text-xs">Refund Date</th>
                                    <th class="text-center text-xs">Amount</th>
                                    <th class="text-center text-xs">Payment Method</th>
                                </tr>
                            </thead>
                            <tbody class="refund-table-body">
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
        document.addEventListener('DOMContentLoaded', function () {
            initTooltips();
            setupEventHandlers();
        });

        function initTooltips() {
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));

            tooltipTriggerList.forEach(function (tooltipTriggerEl) {
                const tooltip = new bootstrap.Tooltip(tooltipTriggerEl, {
                    trigger: 'hover focus',
                    placement: 'top',
                    delay: { show: 100, hide: 50 }
                });

                tooltipTriggerEl.addEventListener('mouseleave', function () {
                    tooltip.hide();
                });
            });

            document.body.addEventListener('mouseover', function (e) {
                const target = e.target.closest('[data-bs-toggle="tooltip"]');
                if (target && !target._tooltip) {
                    const tooltip = new bootstrap.Tooltip(target, {
                        trigger: 'hover focus',
                        placement: 'top'
                    });
                    target._tooltip = tooltip;

                    target.addEventListener('mouseleave', function () {
                        tooltip.hide();
                    });
                }
            });
        }

        function previewImage(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function (e) {
                    document.getElementById('imagePreview').src = e.target.result;
                }
                reader.readAsDataURL(input.files[0]);
            }
        }

        function setupEventHandlers() {
            document.querySelectorAll('.edit-details').forEach(button => {
                button.addEventListener('click', function (e) {
                    const data = JSON.parse(this.getAttribute('data-details'));
                    populateEditModal(data);
                });
            });

            document.querySelectorAll('.view-refund').forEach(button => {
                button.addEventListener('click', function (e) {
                    const data = JSON.parse(this.getAttribute('data-details'));
                    populateRefundModal(data);
                });
            });

            document.querySelector('.details-save')?.addEventListener('click', saveProductDetails);
        }
        function populateEditModal(details) {
            document.querySelector('.detail_id').value = details.product_details_sid;
            document.querySelector('.detail_code').value = details.product_details_code;
            document.querySelector('#purchased_qty').value = details.qty;
            document.querySelector('#supplier_dtls').value =
                `${details.suppliers?.supplier_name || ''} - ${details.suppliers?.phone || ''}`;

            // Store original values for comparison
            document.querySelector('#purchased_qty').dataset.originalQty = details.qty;
            document.querySelector('.product_qty').textContent = details.qty ?? '';
            document.querySelector('.margin_price').textContent = details.margin_price ?? '';

            let totalPaid = 0;
            let refund = 0;
            let loss = 0;

            details.payments.forEach(payment => {
                if (payment.type === 'paid') {
                    totalPaid += parseFloat(payment.paid_amount ?? 0);
                } else if (payment.type === 'refund') {
                    refund += parseFloat(payment.paid_amount ?? 0);
                } else if (payment.type === 'loss') {
                    loss += parseFloat(payment.paid_amount ?? 0);
                }
            });

            const netPaidAmount = totalPaid - refund - loss;
            document.querySelector('.paid_amount').textContent = netPaidAmount.toFixed(2);
            document.querySelector('.updated_qty').textContent = '';
            document.querySelector('.amount').value = '';

            const quantityInput = document.querySelector('#purchased_qty');
            const reasonSelect = document.querySelector('select[name="status"]');
            const amountInput = document.querySelector('.amount');
            const marginPrice = parseFloat(details.margin_price) || 0;

            function updateFields() {
                const originalQty = parseFloat(quantityInput.dataset.originalQty) || 0;
                const newQty = parseFloat(quantityInput.value) || 0;
                const difference = newQty - originalQty;
                const selectedReason = reasonSelect.value;

                // Update quantity difference display
                const updatedQtyCell = document.querySelector('.updated_qty');
                if (difference !== 0) {
                    const sign = difference > 0 ? '+' : '-';
                    const color = difference > 0 ? 'text-success' : 'text-danger';
                    updatedQtyCell.innerHTML = `<span class="${color}">${sign}${Math.abs(difference)}</span>`;
                } else {
                    updatedQtyCell.textContent = '';
                }

                // Reset amount field
                amountInput.value = '';
                const qtyChange = Math.abs(difference);
                const calculatedAmount = qtyChange * marginPrice;

                if (selectedReason === 'update' && difference > 0) {
                    // For quantity increase
                    amountInput.value = calculatedAmount.toFixed(2);
                } else if (selectedReason === 'refund' && difference < 0) {
                    // Refund logic for reduced quantity
                    const returnedQty = originalQty - newQty;

                    const paidQty = netPaidAmount / marginPrice;
                    const unpaidQty = originalQty - paidQty;

                    if (returnedQty > unpaidQty) {
                        const refundableQty = returnedQty - unpaidQty;
                        const refundAmount = refundableQty * marginPrice;

                        if (refundAmount > 0) {
                            amountInput.value = refundAmount.toFixed(2);
                        }
                    }
                } else if (selectedReason === 'non_returnable' && difference < 0) {
                    // Show full lost margin amount
                    amountInput.value = calculatedAmount.toFixed(2);
                }
            }

            quantityInput.addEventListener('input', updateFields);
            reasonSelect.addEventListener('change', updateFields);
        }



        function populateRefundModal(data) {
            const refundTableBody = document.querySelector('.refund-table-body');
            const modalProductName = document.querySelector('.product-name');

            refundTableBody.innerHTML = '';
            modalProductName.textContent = data.name || '';

            if (data.payments && Array.isArray(data.payments)) {
                const refundPayments = data.payments.filter(payment => payment.type === 'refund');

                if (refundPayments.length > 0) {
                    let totalAmount = 0;

                    refundPayments.forEach(payment => {
                        const amount = parseFloat(payment.paid_amount) || 0;
                        totalAmount += amount;

                        const row = `
                                                                                                                                                                                                <tr>
                                                                                                                                                                                                    <td class="text-center text-xs">${payment.payment_date || ''}</td>
                                                                                                                                                                                                    <td class="text-center text-xs">${amount.toFixed(2)}</td>
                                                                                                                                                                                                    <td class="text-center text-xs">${payment.payment_type || ''}</td>
                                                                                                                                                                                                </tr>
                                                                                                                                                                                            `;
                        refundTableBody.insertAdjacentHTML('beforeend', row);
                    });

                    // Add total row
                    const totalRow = `
                                                                                                                                                                                            <tr class="fw-bold">
                                                                                                                                                                                                <td class="text-end text-xs" colspan="1">Total</td>
                                                                                                                                                                                                <td class="text-center text-xs">${totalAmount.toFixed(2)}</td>
                                                                                                                                                                                                <td></td>
                                                                                                                                                                                            </tr>
                                                                                                                                                                                        `;
                    refundTableBody.insertAdjacentHTML('beforeend', totalRow);
                } else {
                    refundTableBody.innerHTML = `
                                                                                                                                                                                            <tr>
                                                                                                                                                                                                <td colspan="3" class="text-center text-xs">No Refunds Found</td>
                                                                                                                                                                                            </tr>
                                                                                                                                                                                        `;
                }
            } else {
                refundTableBody.innerHTML = `
                                                                                                                                                                                        <tr>
                                                                                                                                                                                            <td colspan="3" class="text-center text-xs">No Refunds Found</td>
                                                                                                                                                                                        </tr>
                                                                                                                                                                                    `;
            }
        }

        function saveProductDetails() {
            const detailsForm = document.querySelector('.product-details-form');

            if (detailsForm.checkValidity()) {
                const formData = new FormData(detailsForm);

                fetch('{{ route('product-details-update') }}', {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                title: 'Success!',
                                text: data.message || 'Product details updated successfully.',
                                icon: 'success',
                                confirmButtonText: 'OK'
                            }).then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire({
                                title: 'Error!',
                                text: data.message || 'An error occurred while updating the product details.',
                                icon: 'error',
                                confirmButtonText: 'OK'
                            });
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        Swal.fire({
                            title: 'Error!',
                            text: 'An error occurred while updating the product details.',
                            icon: 'error',
                            confirmButtonText: 'OK'
                        });
                    });
            } else {
                detailsForm.reportValidity();
            }
        }
    </script>
@endpush